/*
    Xclones - A game of skill.  Try to find the Intellivision Running Man
    clones under your windows.
    
    Copyright 1991 by J.T. Anderson

    jta@locus.com
    
    This program may be freely distributed provided that all
    copyright notices are retained.

    To build:
      cc -o xclones xclones.c -lX11 [-lsocketorwhatever] [-lm] [-l...]

    Dedicated to Greg McFarlane.   (gregm@otc.otca.oz.au)
    
    Squish option contributed by Rick Petkiewizc (rick@locus.com)

    Virtual root code adapted from patch sent by Colin Rafferty who
    borrowed it from Tom LaStrange.  Several other folks sent similar
    fixes.

    Converted from xroach to xclones by Joe Zbiciak (im14u2c@primenet.com)
    
*/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include <signal.h>

#include <limits.h>

#if __STDC__
#include <stdlib.h>
#else
long strtol();
double strtod();
char *getenv();
#endif

char Copyright[] = "Xclones (a JZ hack, 1999)\nBased on Xroach\n"
                   "Original Copyright 1991 J.T. Anderson\n";

#include "clonemap.h"

typedef unsigned long Pixel;
typedef int ErrorHandler();

#define SCAMPER_EVENT   (LASTEvent + 1)

#if !defined(GRAB_SERVER)
#define GRAB_SERVER     0
#endif

Display *display;
int screen;
Window rootWin;
unsigned int display_width, display_height;
int center_x, center_y;
GC gc;
GC gutsGC;
char *display_name = NULL;
Pixel black, white;

int done = 0;
int eventBlock = 0;
int errorVal = 0;
Bool squishClone = False;
Pixmap squish0Map;
Pixmap squish1Map;
Bool squishWinUp = False;

typedef struct Clone {
    CloneMap *rp;
    CloneMap *frame;
    int index;
    float x;
    float y;
    int intX;
    int intY;
    int hidden;
    int under;
    int turnLeft;
    int steps;
} Clone;

Clone *clones;
int maxClones = 8;
int curClones = 0;
float cloneSpeed = 20.0;

Region rootVisible = NULL;

void Usage();
void SigHandler();
void AddClone();
void MoveClone();
void DrawClones();
void CoverRoot();
int CalcRootVisible();
int MarkHiddenClones();
Pixel AllocNamedColor();
Window FindRootWindow();

void
main(ac, av)
int ac;
char *av[];
{
    XGCValues xgcv;
    int ax;
    char *arg;
    CloneMap *rp;
    int rx;
    float angle;
    XEvent ev;
    char *cloneColor = "#00CCFF";
    char *gutsColor = NULL;
    int nVis;
    int needCalc;
    Window squishWin;
    XSetWindowAttributes xswa;
    
    /*
       Process command line options.
    */
    for (ax=1; ax<ac; ax++) {
        arg = av[ax];
        if (strcmp(arg, "-display") == 0) {
            display_name = av[++ax];
        }
        else if (strcmp(arg, "-rc") == 0) {
            cloneColor = av[++ax];
        }
        else if (strcmp(arg, "-speed") == 0) {
            cloneSpeed = strtod(av[++ax], (char **)NULL);
        }
        else if (strcmp(arg, "-clones") == 0) {
            maxClones = strtol(av[++ax], (char **)NULL, 0);
        }
        else if (strcmp(arg, "-squish") == 0) {
            squishClone = True;
        }
        else if (strcmp(arg, "-rgc") == 0) {
            gutsColor = av[++ax];
        }
        else {
            Usage();
        }
    }

#ifdef NO_DRAND48
    srand((int)time((long *)NULL));
#else
    srand48((int)time((long *)NULL));
#endif
    
    /*
       Catch some signals so we can erase any visible clones.
    */
    signal(SIGKILL, SigHandler);
    signal(SIGINT, SigHandler);
    signal(SIGTERM, SigHandler);
    signal(SIGHUP, SigHandler);

    display = XOpenDisplay(display_name);
    if (display == NULL) {
        if (display_name == NULL) display_name = getenv("DISPLAY");
        (void) fprintf(stderr, "%s: cannot connect to X server %s\n", av[0],
            display_name ? display_name : "(default)");
        exit(1);
    }

    screen = DefaultScreen(display);
    rootWin = FindRootWindow();
    black = BlackPixel(display, screen);
    white = WhitePixel(display, screen);

    display_width = DisplayWidth(display, screen);
    display_height = DisplayHeight(display, screen);
    center_x = display_width / 2;
    center_y = display_height / 2;
    
    /*
       Create clone pixmaps at several orientations.
    */
    for (rx = 0; rx < CLONE_HEADINGS ; rx++) {
        angle = rx * 2. * M_PI / (double) CLONE_HEADINGS;
        rp = &clonePix[rx];
        rp->sine = sin(angle);
        rp->cosine = cos(angle);
    }

    for (rx = 0; rx < sizeof(clonePix) / sizeof(CloneMap); rx++) {
        rp = &clonePix[rx];
        rp->pixmap = XCreateBitmapFromData(display, rootWin,
            rp->cloneBits, rp->width, rp->height);
    } 

    /*
      Create the squished pixmap
    */
    if (squishClone) {
        squish0Map = XCreateBitmapFromData(display, rootWin,
                     squish0_bits, squish0_width, squish0_height);
        squish1Map = XCreateBitmapFromData(display, rootWin,
                     squish1_bits, squish1_width, squish1_height);
    }

    clones = (Clone *)malloc(sizeof(Clone) * maxClones);

    gc = XCreateGC(display, rootWin, 0L, &xgcv);
    XSetForeground(display, gc, AllocNamedColor(cloneColor, black));
    XSetFillStyle(display, gc, FillStippled);

    if (squishClone && gutsColor != NULL) {
        gutsGC = XCreateGC(display, rootWin, 0L, &xgcv);
        XSetForeground(display, gutsGC, AllocNamedColor(gutsColor, black));
        XSetFillStyle(display, gutsGC, FillStippled);
    }
    else
        gutsGC = gc;
    
    while (curClones < maxClones)
        AddClone();
    
    XSelectInput(display, rootWin, ExposureMask | SubstructureNotifyMask);

    if (squishClone) {
        xswa.event_mask = ButtonPressMask;
        xswa.override_redirect = True;
        squishWin = XCreateWindow(display, rootWin, 0, 0,
                        display_width, display_height, 0,
                        CopyFromParent, InputOnly, CopyFromParent,
                        CWOverrideRedirect | CWEventMask, &xswa);
        XLowerWindow(display, squishWin);
    }
    
    needCalc = 1;
    while (!done) {
        if (XPending(display)) {
            XNextEvent(display, &ev);
        }
        else {
            if (needCalc) {
                needCalc = CalcRootVisible();
            }
            if (needCalc)
                nVis = 0;
            else
                nVis = MarkHiddenClones();
            if (nVis) {
                ev.type = SCAMPER_EVENT;
                if (!squishWinUp && squishClone) {
                    XMapWindow(display, squishWin);
                    squishWinUp = True;
                }
            }
            else {
                if (squishWinUp && squishClone) {
                    XUnmapWindow(display, squishWin);
                    squishWinUp = False;
                }
                if (needCalc == 0)
                    DrawClones();
                eventBlock = 1;
                XNextEvent(display, &ev);
                eventBlock = 0;
            }
        }
        
        switch (ev.type) {
            
            case SCAMPER_EVENT:
                for (rx=0; rx<curClones; rx++) {
                    if (!clones[rx].hidden)
                        MoveClone(rx);
                }
                DrawClones();
                XSync(display, False);
#ifndef NO_USLEEP
		usleep(10000);
#endif
                break;
                
            case UnmapNotify:
                if (ev.xunmap.window != squishWin)
                    needCalc = 1;
                break;

            case MapNotify:
            case Expose:
            case ConfigureNotify:
                needCalc = 1;
                break;
        
            case ButtonPress:
                checkSquish(&ev);
                break;
                
        }
    	if (RandInt(20000) < 10 && curClones < maxClones)
       	    AddClone();
    }
    
    CoverRoot();
    
    XCloseDisplay(display);
}

#define USEPRT(msg) fprintf(stderr, msg)

void
Usage()
{
    USEPRT("Usage: xclone [options]\n\n");
    USEPRT("Options:\n");
    USEPRT("       -display displayname\n");
    USEPRT("       -rc      clonecolor\n");
    USEPRT("       -clones numclones\n");
    USEPRT("       -speed   clonespeed\n");
    USEPRT("       -squish\n");
    USEPRT("       -rgc     clonegutscolor\n");
    
    exit(1);
}

void
SigHandler()
{
       
    /*
       If we are blocked, no clones are visible and we can just bail
       out.  If we are not blocked, then let the main procedure clean
       up the root window.
    */
    if (eventBlock) {
        XCloseDisplay(display);
        exit(0);
    }
    else {
        done = 1;
    }
}

/*
   Find the root or virtual root window.
*/
Window
FindRootWindow()
{
    Window rootWin;
    Window realRoot;
    Atom swmVroot;
    Window rootReturn, parentReturn, *children;
    unsigned int numChildren;
    int cx;
    Atom actualType;
    Atom actualFormat;
    unsigned long nItems;
    unsigned long bytesAfter;
    Window *newRoot;
    
    /*
       Get real root window.
    */
    realRoot = rootWin = RootWindow(display, screen);
    
    /*
       Get atom for virtual root property.  If the atom doesn't
       exist, we can assume the corresponding property does not
       exist. 
    */
    swmVroot = XInternAtom(display, "__SWM_VROOT", True);
    
    if (swmVroot == None)
        return rootWin;
    
    /*
       Run children of root, looking for a virtual root.
    */
    XQueryTree(display, rootWin, &rootReturn, &parentReturn, 
                    &children, &numChildren);
    for (cx=0; cx<numChildren; cx++) {
        newRoot = NULL;
        nItems = 0;
        if (XGetWindowProperty(display, children[cx], swmVroot, 0L, 1L, False,
            XA_WINDOW, &actualType, &actualFormat, &nItems,
            &bytesAfter, &newRoot) == Success && actualFormat != None) {
                if (nItems >= 1) {
                    rootWin = *newRoot;
                }
                if (newRoot) {
                    XFree(newRoot);
                }
        }
        if (rootWin != realRoot) break;
    }
    XFree(children);
    
    return rootWin;
}

/*
   Generate random integer between 0 and maxVal-1.
*/
int
RandInt(maxVal)
int maxVal;
{
#ifdef NO_DRAND48
        return rand() * (double)maxVal/ (double)RAND_MAX;
#else
	return drand48() * maxVal;
#endif
}

/*
   Check for clone completely in specified rectangle.
*/
int
CloneInRect(clone, rx, ry, x, y, width, height)
Clone *clone;
int rx;
int ry;
int x;
int y;
unsigned int width;
unsigned int height;
{
    if (rx < x) return 0;
    if ((rx + clone->rp->width) > (x + width)) return 0;
    if (ry < y) return 0;
    if ((ry + clone->rp->height) > (y + height)) return 0;
    
    return 1;
}

/*
   Check for clone overlapping specified rectangle.
*/
int
CloneOverRect(clone, rx, ry, x, y, width, height)
Clone *clone;
int rx;
int ry;
int x;
int y;
unsigned int width;
unsigned int height;
{
    if (rx >= (x + width)) return 0;
    if ((rx + clone->rp->width) <= x) return 0;
    if (ry >= (y + height)) return 0;
    if ((ry + clone->rp->height) <= y) return 0;
    
    return 1;
}

/*
   Give birth to a clone.
*/
void
AddClone()
{
    Clone *r;
    
    if (curClones < maxClones) {
        r = &clones[curClones++];
        r->index = RandInt(CLONE_HEADINGS);
        r->rp = &clonePix[r->index];
        r->x = RandInt(display_width - r->rp->width);
        r->y = RandInt(display_height - r->rp->height);
        r->intX = -1;
        r->intY = -1;
        r->hidden = 0;
        r->steps = RandInt(200);
        r->turnLeft = RandInt(100) >= 50;
    }
}

/*
   Turn a clone.
*/
void
TurnClone(clone)
Clone *clone;
{
    if (clone->index != (clone->rp - clonePix)) return;

    if (clone->turnLeft) {
        clone->index += (RandInt(30) / 10) + 1;
        if (clone->index >= CLONE_HEADINGS)
            clone->index -= CLONE_HEADINGS;
    }
    else {
        clone->index -= (RandInt(30) / 10) + 1;
        if (clone->index < 0)
            clone->index += CLONE_HEADINGS;
    }
}

/*
   Move a clone.
*/
void
MoveClone(rx)
int rx;
{
    Clone *clone;
    Clone *r2;
    float newX;
    float newY;
    int ii;
    
    clone = &clones[rx];
    newX = clone->x + (cloneSpeed * clone->rp->cosine);
    newY = clone->y - (cloneSpeed * clone->rp->sine);
    
    if (CloneInRect(clone, (int)newX, (int)newY, 
                            0, 0, display_width, display_height)) {
        
        clone->x = newX;
        clone->y = newY;

        if (clone->steps-- <= 0) {
            TurnClone(clone);
            clone->steps = RandInt(200);
        }

        for (ii=rx+1; ii<curClones; ii++) {
            r2 = &clones[ii];
            if (CloneOverRect(clone, (int)newX, (int)newY,
                r2->intX, r2->intY, r2->rp->width, r2->rp->height)) {
        
                TurnClone(clone);
            }
        }
    }
    else {
        TurnClone(clone);
    }
}
    
/*
   Draw all clones.
*/
void
DrawClones()
{
    Clone *clone;
    int rx;
   
#if 0 
    for (rx=0; rx<curClones; rx++) {
        clone = &clones[rx];
        
        if (clone->intX >= 0 && clone->rp != NULL) {
            XClearArea(display, rootWin, clone->intX, clone->intY,
                clone->rp->width, clone->rp->height, False);
        }
    }
#endif
    
    for (rx=0; rx<curClones; rx++) {
	int oldX, oldY, fr; 
	CloneMap *old_frame;

        clone = &clones[rx];

	oldX = clone->intX;
	oldY = clone->intY;
	old_frame = clone->frame;

	if (oldX >= 0 && old_frame) { 
	    int ii;

	    XClearArea(display, rootWin, oldX, oldY,
		       old_frame->width, old_frame->height, False);

            for (ii = 0; ii < rx; ii++) {
            	Clone *r2 = &clones[rx];
		if (r2->intX >= 0 && r2->frame &&
                    CloneOverRect(clone, (int)oldX, (int)oldY,
                    r2->intX, r2->intY, r2->frame->width, r2->frame->height)) {
		    r2->under = 1;
		}
            }
	}

        if (!clone->hidden) {
            int ofs; 

            clone->intX = 6 * (fr = (int)clone->x/6);
            clone->intY = 3 * ((int)clone->y/3);
            clone->rp = &clonePix[clone->index];

            /* select actual pixmap depending on x, y and dir. */
            if (fabs(clone->rp->sine) > 1.2 * fabs(clone->rp->cosine))
            {
                ofs = CLONE_VERT;  /* vertical */
		fr  = clone->intY / 12;
            } else if (clone->rp->cosine < 0) { 
                ofs = CLONE_LEFT;  /* facing left */
            } else {
                ofs = CLONE_RIGHT; /* facing right */
            } 
            clone->frame = &clonePix[ofs + fr % CLONE_FRAMES];
    
            XSetStipple(display, gc, clone->frame->pixmap);
            XSetTSOrigin(display, gc, clone->intX, clone->intY);
            XFillRectangle(display, rootWin, gc,
                clone->intX, clone->intY, 
		clone->frame->width, clone->frame->height);
        }
        else {
            clone->intX = -1;
        }
    }
    for (rx=0; rx<curClones; rx++) {
        clone = &clones[rx];

	if (clone->under && clone->intX > 0 && clone->frame) {
            XSetStipple(display, gc, clone->frame->pixmap);
            XSetTSOrigin(display, gc, clone->intX, clone->intY);
            XFillRectangle(display, rootWin, gc,
                clone->intX, clone->intY, 
		clone->frame->width, clone->frame->height);
	}
        clone->under = 0;
    }
}

/*
   Cover root window to erase clones.
*/
void
CoverRoot()
{
    XSetWindowAttributes xswa;
    long wamask;
    Window cloneWin;
    
    xswa.background_pixmap = ParentRelative;
    xswa.override_redirect = True;
    wamask = CWBackPixmap | CWOverrideRedirect;
    cloneWin = XCreateWindow(display, rootWin, 0, 0,
                    display_width, display_height, 0, CopyFromParent,
                    InputOutput, CopyFromParent, wamask, &xswa);
    XLowerWindow(display, cloneWin);
    XMapWindow(display, cloneWin);
    XFlush(display);
}    

#if !GRAB_SERVER

int
CloneErrors(dpy, err)
Display *dpy;
XErrorEvent *err;
{
    errorVal = err->error_code;
    
    return 0;
}

#endif /* GRAB_SERVER */

/*
   Calculate Visible region of root window.
*/
int
CalcRootVisible()
{
    Region covered;
    Region visible;
    Window *children;
    int nChildren;
    Window dummy;
    XWindowAttributes wa;
    int wx;
    XRectangle rect;
    int winX, winY;
    unsigned int winHeight, winWidth;
    unsigned int borderWidth;
    unsigned int depth;
    
    /*
       If we don't grab the server, the XGetWindowAttribute or XGetGeometry
       calls can abort us.  On the other hand, the server grabs can make for
       some annoying delays.
    */
#if GRAB_SERVER
    XGrabServer(display);
#else
    XSetErrorHandler(CloneErrors);
#endif

    /*
       Get children of root.
    */
    XQueryTree(display, rootWin, &dummy, &dummy, &children, &nChildren);
    
    /*
       For each mapped child, add the window rectangle to the covered
       region.
    */
    covered = XCreateRegion();
    for (wx=0; wx<nChildren; wx++) {
        if (XEventsQueued(display, QueuedAlready)) {
            XDestroyRegion(covered);
            return 1;
        }
        errorVal = 0;
        XGetWindowAttributes(display, children[wx], &wa);
        if (errorVal) continue;
        if (wa.class == InputOutput && wa.map_state == IsViewable) {
            XGetGeometry(display, children[wx], &dummy, &winX, &winY,
                &winWidth, &winHeight, &borderWidth, &depth);
            if (errorVal) continue;
            rect.x = winX;
            rect.y = winY;
            rect.width = winWidth + (borderWidth * 2);
            rect.height = winHeight + (borderWidth * 2);
            XUnionRectWithRegion(&rect, covered, covered);
        }
    }
    XFree(children);

#if GRAB_SERVER
    XUngrabServer(display);
#else
    XSetErrorHandler((ErrorHandler *)NULL);
#endif
    
    /*
       Subtract the covered region from the root window region.
    */
    visible = XCreateRegion();
    rect.x = 0;
    rect.y = 0;
    rect.width = display_width;
    rect.height = display_height;
    XUnionRectWithRegion(&rect, visible, visible);
    XSubtractRegion(visible, covered, visible);
    XDestroyRegion(covered);
    
    /*
       Save visible region globally.
    */
    if (rootVisible)
        XDestroyRegion(rootVisible);
    rootVisible = visible;
    
    
    /*
       Mark all clones visible.
    */
    for (wx=0; wx<curClones; wx++) 
        clones[wx].hidden = 0;

    return 0;
}

/*
   Mark hidden clones.
*/
int
MarkHiddenClones()
{
    int rx;
    Clone *r;
    int nVisible;
    
    nVisible = 0;
    for (rx=0; rx<curClones; rx++) {
        r = &clones[rx];
        
        if (!r->hidden) {
            if (r->intX > 0 && XRectInRegion(rootVisible, r->intX, r->intY,
                            r->rp->width, r->rp->height) == RectangleOut) {
                r->hidden = 1;
            }
            else {
                nVisible++;
            }
        }
    }
    
    return nVisible;
}

/*
   Allocate a color by name.
*/
Pixel
AllocNamedColor(colorName, dfltPix)
char *colorName;
Pixel dfltPix;
{
        Pixel pix;
        XColor scrncolor;
        XColor exactcolor;

        if (XAllocNamedColor(display, DefaultColormap(display, screen),
                colorName, &scrncolor, &exactcolor)) {
                pix = scrncolor.pixel;
        }
        else {
                pix = dfltPix;
        }

        return pix;
}

/*
 *      squishCheck - Check to see if we have to squish any clones.
 */
checkSquish(buttonEvent)
XButtonEvent *buttonEvent;
{
    int x, y;
    int i;
    int rx;
    Clone *r;
/* */
    x = buttonEvent->x;
    y = buttonEvent->y;

    for (rx=0; rx<curClones; rx++) {
        r = &clones[rx];
        if (r->rp == NULL) continue;

        if (x > r->intX &&
            x < (r->intX + r->rp->width) &&
            y > r->intY &&
            y < (r->intY + r->rp->height)) {

	    if (r->intX >= 0 && r->frame)
            	XClearArea(display, rootWin, r->intX, r->intY,
               		   r->frame->width, r->frame->height, False);
            if ( (RandInt(128) > 64) ) {
                XSetStipple(display, gutsGC, squish0Map);
                XSetTSOrigin(display, gutsGC, r->intX, r->intY);
                XFillRectangle(display, rootWin, gutsGC,
                    r->intX, r->intY, squish0_width, squish0_height);
            } else {
                XSetStipple(display, gutsGC, squish1Map);
                XSetTSOrigin(display, gutsGC, r->intX, r->intY);
                XFillRectangle(display, rootWin, gutsGC,
                    r->intX, r->intY, squish1_width, squish1_height);
            }
            /*
             * Delete the clone
             */
            for (i = rx; i < curClones - 1; i++)
                clones[i] = clones[i + 1];
        
            curClones--;
            rx--;
        }
    }

    return;
}
